# UNITENGINE
# Contents:
#  UnitEngine styles
#  UnitEngine class definition
#  Unit class definition

init -9 python:
  # Styles used by the UnitEngine
  # These can be overridden in any init block at -8 or higher
  SemiTransparentGrey = Solid((150, 150, 150, 128))
  style.engine_frame["buttons"].xfill=False
  style.engine_frame["buttons"].yminimum=None
  style.engine_frame["buttons"].xalign=1.0
  style.engine_frame["buttons"].ypos=0
  style.engine_frame["buttons"].yanchor=0.0
  style.engine_frame["buttons"].background=SemiTransparentGrey
  style.engine_frame["buttons"].right_margin=20

  style.engine_frame["menu"].take(style.engine_frame["buttons"])
  style.engine_frame["unit info"].take(style.engine_frame["buttons"])
  style.engine_frame["unit info"].ypos=200

  style.engine_label["unit name"].xalign=1.0
  style.engine_label_text["unit name"].size=30
  style.engine_menu.xalign=0.9
  style.engine_menu.yalign=0.0

  #######################################
  ### Definition of UnitEngine object ###
  #######################################

  class UnitEngine(TileEngine):
    """Engine to display units on a tile map and move them around in turns.
    """
    # Constants that're available before instantiation:
    # Movement styles
    MP = 1
    AP = 2

    # Constant modes
    NOSELECTION   = 1
    ENEMYSELECTED = 4
    UNITSELECTED  = 8
    MOVEMODE      = 9
    ACTMODE       = 10
    ENDOFTURN     = -1
    VICTORY       = -2
    DEFEAT        = -3

    # Constants for two-player games
    HUMAN = 0
    COMPUTER = 1
    PLAYER = Struct(name = "Player", control = HUMAN, HP_scale = 1.0)
    ENEMY = Struct(name = "Enemy", control = COMPUTER, HP_scale = 1.0)

    # Terrain types
    TT_GROUND = Struct(MoveableThrough = True, MoveCost = 1)
    TT_WALL   = Struct(MoveableThrough = False, MoveCost = 0)

    def __init__(self,
        movement_style = None,
        move_square_highlight = ("movehighlight.png","movehighlight.png"),
        act_square_highlight = ("acthighlight.png","acthighlight.png"),
        unit_highlight = None,
        unit_highlight_function = Image,
        player_list = [PLAYER, ENEMY],
        confirm_end_turn = False,
        end_turn_when_finished = True,
        end_when_all_enemies_dead = True,
        allow_moving_through_units = True,
        movement_directions = 8,
        display_directions = None,
        show_unit_info = True,
        unit_info = None,
        show_buttons = True,
        moving_sprite_status = "moving",
        keymap = None,
        computer_turn_function = None,
        **properties):
      """
      Create the UnitEngine. Parameters:
        movement_style: must be UnitEngine.MP or UnitEngine.AP. In this release, only MP mode is supported.
        move_square_highlight: A filename or (idle, hover) tuple of filenames. The first filename is the image to display on squares to which the selected unit can move. The second, if present, is the image for such squares when the mouse is hovered over them. The images should probably be semitransparent.
        act_square_highlight: A filename or tuple of two filenames, as for ''move_square_highlight'', but used for default highlighting in unit actions. This value is used in HighlightSquares if no highlight is specified in that function.
        unit_highlight: A filename for an image, which is displayed underneath the currently selected unit.
        unit_highlight_function: The function to display the ''unit_highlight'', which must accept position properties - xpos, ypos, xanchor and yanchor. The default is Image.
        player_list: A list of the players in this game, in the order they'll take their turns. Default is [UnitEngine.PLAYER, UnitEngine.ENEMY]. Each entry must be an object with properties ''name'', ''control'' and ''HP_scale''. Two predefined Players are provided: UnitEngine.PLAYER and UnitEngine.ENEMY. The ''control'' property of each object must be either UnitEngine.HUMAN or UnitEngine.COMPUTER. The ''HP_scale'' property is a floating-point number, by which the HP of all the player's units will be multiplied. This can provide a simple way of scaling difficulty.
        confirm_end_turn: If True, after the player selects End Turn then an confirmation menu "End Turn" / "Cancel" is presented. If False (default), the turn will end without this confirmation menu.
        end_turn_when_finished: By default, when the player's last unit is finished moving and acting, the turn will end automatically. If this flag is False, instead the player must always select End Turn manually.
        end_when_all_enemies_dead: By default, the game will end when all units left on the map are controlled by the player. If you want the game to continue in such circumstances (or the game without any non-player-controlled units), set this flag to False.
        allow_moving_through_units: By default, units can't end their movement in the same space another unit is in, but they can move through other units. If this flag is False, units will never be allowed to pass through a square another unit's in.
        movement_directions: This parameter must be either 4 or 8. If it's 8 (the default), all units can move diagonally or orthogonally. If it's 4, only orthogonal movement is allowed: nothing moves diagonally.
        display_directions: This parameter must be either 1, 4 or 8, or the special value None, meaning to inherit from ''movement_directions'', which is the default. If it's 8, units are assumed to have eight variant images for each unit status, one for each of the directions N, NE, E, SE, S, SW, W and NW. The ''direction_facing'' property of a unit may be any one of the parent UnitEngine's properties DIR_N, DIR_NW, etc. If this parameter is 4, units will only face in the four orthogonal directions (NW, SW, NE and SE for isometric; N, S, E and W for straight.) If this parameter is 1, units will not have their ''direction_facing'' property changed. Note that units will be created with their ''display_directions'' equal to the engine's ''display_directions'', but it may be overridden on individual units.
        show_unit_info: If this flag is True (the default), an overlay will be displayed giving information about the current selected unit, in style engine_frame["unit info"]. The information can be specified in ''unit_info''.
        unit_info: This may be None, meaning use some default unit information; or it may be a list of info items. Each info item is either a function, which should add a line of unit information to the info panel, or a (conditionfunction, infofunction) tuple, in which case the infofunction will only be called if the conditionfunction evaluates to True.
        show_buttons: By default, the UnitEngine will show buttons to let the player instruct units to Move, Act, and so on. Set this parameter to False to hide these buttons. Note that the keyboard shortcuts are controlled separately by the ''keymap'' parameter.
        keymap: A dictionary mapping keysyms to functions. The functions must be wrapped in ui.returns() or something similar, as the result of calling the function will itself be called after ui.interact finishes. If this is None, the default keymap is used. To specify an empty keymap (disable keyboard control), use {}.
        moving_sprite_status: A string that will be inserted as the status field into unit imagenames when units move. Default is "moving".
        computer_turn_function: A function that will be called when the current player has their ''controller'' property set to UnitEngine.COMPUTER. Default is e.AI_TakeTurn, which is a basic AI function.
      """

      # Require the user to specify the movement_style
      assert movement_style in [self.MP, self.AP], 'UnitEngine() must be called with a movement_style equal to UnitEngine.MP or UnitEngine.AP !'
      self.movement_style = movement_style

      # Handle other supplied properties
      if type(move_square_highlight) != type( (0,0) ):
        move_square_highlight = (move_square_highlight, move_square_highlight)
      self.move_square_highlight = move_square_highlight
      if type(act_square_highlight) != type( (0,0) ):
        act_square_highlight = (act_square_highlight, act_square_highlight)
      self.act_square_highlight = act_square_highlight
      self.player_list = player_list
      self.confirm_end_turn = confirm_end_turn
      self.end_turn_when_finished = end_turn_when_finished
      self.end_when_all_enemies_dead = end_when_all_enemies_dead
      self.allow_moving_through_units = allow_moving_through_units
      self.moving_sprite_status = moving_sprite_status
      assert movement_directions in [4, 8], "movement_directions can only be 4 or 8"
      self.movement_directions = movement_directions
      if display_directions == None:
        display_directions = movement_directions
      assert display_directions in [1, 4, 8], "display_directions can only be 1, 4 or 8, or None to equal movement_directions"
      self.display_directions = display_directions

      self.computer_turn_function = computer_turn_function or self.AI_TakeTurn


      # Handle any supplied properties
      TileEngine.__init__(self, **properties)

      # Assign default values to more properties
      self.units = [ ]
      self.selected_unit = None
      self.stored_movement_costs_for_unit = None
      self.menu_items = None
      self.is_in_interaction = False

      self._mode = self.NOSELECTION

      # Highlights
      if unit_highlight == None:
        self.unit_highlight_sprite = None
      else:
        self.unit_highlight_sprite = self.Sprite(sprite_name=unit_highlight, function=unit_highlight_function, position=(None,None), visible=False)

      # Define the default callbacks
      self.callbacks.mode_change = None
      self.callbacks.unit_moves = None # Parameters: unit, position
      self.callbacks.unit_acts = None
      self.callbacks.square_show_pre_sprite = self.SquareShowHighlights
      # Leave self.callbacks.square_show_post_sprite as it is
      self.callbacks.unit_damaged = None
      self.callbacks.unit_destroyed = None

      self.show_unit_info = show_unit_info
      self.show_buttons = show_buttons
      if unit_info:
        self.unit_info = unit_info
      else:
        # Define the default overlay
        if self.movement_style == self.MP:
          self.unit_info = [
                self.ShowUnitName,
                self.ShowUnitHP,
                (self.IsPlayerUnit, self.ShowUnitMP),
                (self.IsPlayerUnit, self.ShowUnitActionsLeft),
                self.ShowUnitPosition
                  ]
        else:
          self.unit_info = [
                self.ShowUnitName,
                self.ShowUnitHP,
                (self.IsPlayerUnit, self.ShowUnitAP),
                self.ShowUnitPosition
                  ]

      # Handle the keymap
      if keymap is None:
        # We've already got the default TileEngine keymap; we want to add to it
        self.keymap.update(dict(
                 K_RETURN = ui.returns(self.ActivateCursorSquare),
               K_KP_ENTER = ui.returns(self.ActivateCursorSquare),
                  K_SPACE = ui.returns(self.ActivateCursorSquare),
                      K_m = ui.returns(self.CurriedSetMode(self, self.MOVEMODE)),
                      K_a = ui.returns(self.CurriedSetMode(self, self.ACTMODE)),
                      K_n = ui.returns(self.SelectNextUnit),
                        E = ui.returns(self.ConfirmEndTurn),
                 K_ESCAPE = ui.returns(self.CancelAction)
                    ))
      else:
        # Overwrite the default TileEngine keymap with the specified one
        self.keymap = keymap

      # Add the overlay functions
      if self.overlay_ShowButtons not in config.overlay_functions:
        config.overlay_functions.append(self.overlay_ShowButtons)
      if self.overlay_ShowUnitInfo not in config.overlay_functions:
        config.overlay_functions.append(self.overlay_ShowUnitInfo)

      # Add fields to the map
      for tx in range(self.gxmax):
        for ty in range(self.gymax):
          self.map[tx][ty].unit = None
          self.map[tx][ty].highlight = None
          self.map[tx][ty].under_units = []

    # Method Unit(): create and return a Unit object
    def Unit(self, **properties):
      """
      Create and return a Unit object, which will be displayed on the UnitEngine's map.
      Parameters:
        name: A string name for this unit. This will be displayed in the default unit_info unless you override it. If not supplied, defaults to the ''sprite_base_name''.
        sprite_base_name: Required. The filename from which the image names will be derived. The image filename used for the unit is derived as follows:
        The ''sprite_base_name'' is split at its last dot ("."). If the ''direction_facing.name'' is nonempty, it will be prefixed by a "-" and inserted before the last dot. Then if the ''sprite_status'' is nonempty, it also will be prefixed by a "-" and inserted before the last dot.
        So for example, if the ''sprite_base_name'' is "knight.png", the ''direction_facing'' is e.DIR_NW, and the ''sprite_status'' is "sword", the the image filename will be "knight-nw-sword.png".

        hover_sprite_base_name: If not None, this is a filename which is used instead of the ''sprite_base_name'' when the mouse cursor is hovering over the unit. The ''direction_facing'' and ''sprite_status'' are inserted as per the ''sprite_base_name''.
        max_HP: The unit's maximum hit points. Defaults to None.
        max_MP: The unit's maximum MP (if the UnitEngine is in MP mode). Defaults to None.
        max_action: The number of actions the unit's allowed to take each turn (if the UnitEngine is in MP mode). Defaults to 1.
        max_A: The unit's maximum AP (if the UnitEngine is in AP mode). Defaults to None.
        direction_facing: The direction the unit starts off facing. Should be one of the UnitEngine's DIR_* constants. Default is DIR_NEUTRAL.
        position: Where on the grid the unit is. Must be a (gridx, gridy) tuple. It can be (None, None), the default, in which case the unit won't be displayed until you change its position to something else.
        can_move: If False, the unit can't enter movement mode. Default is True.
        can_act: If False, the unit can't enter action mode. Default is True.
        controller: A Player struct, which must be a member of the UnitEngine's ''player_list''.
        display_directions: If this unit has more or fewer distinct directions in which images are available, compared to the engine's ''display_directions'', then you can override it on a per-unit basis by setting this property.
        actions: This is a dictionary mapping action names to functions. If this property is None, the "Act" button is disabled. When the player selects "Act", if this dictionary has just one entry in it, that function is called; otherwise, a menu is presented of each of the action names, and when one is selected, the corresponding function is called.
        clicked: The function to be called when the unit is clicked. Two special values are understood: None means the unit does nothing when clicked, and -1 (the default) means to use the default clicked function, which will choose the unit as a target if the ''mode'' is ACTMODE, and otherwise make the clicked unit the selected unit.
        sprite_status: A string that gets inserted into the image filename. See ''sprite_base_name'' for details.
      """
      # Create a Unit object, passing ourself (the UnitEngine) as first arg
      return Unit(self, **properties)

    # Method Show(): show the map and units
    def Show(self, *args, **properties):
      """
      Show the map and units. Changes to map or units will only be displayed when this function is called.
      The same parameters are supported as TileEngine.Show().
      """
      # Update the unit highlight if it exists
      if self.unit_highlight_sprite:
        if self.selected_unit is not None:
          self.unit_highlight_sprite.position = self.selected_unit.position
          self.unit_highlight_sprite.visible = True
        else:
          self.unit_highlight_sprite.visible = False

      # Show the grid
      TileEngine.Show(self, *args, **properties)

    # SquareShowHighlights: a callback in square_show_pre_sprite to show the highlights
    def SquareShowHighlights(self, (gx, gy), (sx, sy)):
      if self.map[gx][gy].highlight is not None:
        ui.at(self.Position((sx, sy)))
        ui.imagebutton(self.map[gx][gy].highlight[0], self.map[gx][gy].highlight[1], clicked=ui.returns(self.ClickSquare(self, (gx,gy))), focus_mask=True)

    def StartGame(self):
      """
      Starts the game sequence, letting players take turns in sequence until the end game conditions are met.
      """
      # Assign all units' HP to their max_HP
      self.ResetHP()
      # Loop taking the players' turns until the game's over
      while (self.mode not in [self.VICTORY, self.DEFEAT]):
        for p in self.player_list:
          self.StartOfTurn(p)
          if self.CheckEndGame():
            break
      # When the game's done, we need to hide the engine view afterwards
      self.Hide()

    def ResetHP(self):
      """Called at the start of the game.
      """
      for thisunit in self.units:
        thisunit.HP = thisunit.max_HP


    def StartOfTurn(self, player):
      """Take a turn for the specified player.
      """
      self.current_player = player
      self.ResetFlagsAndMP()
      self.selected_unit = None
      # Force us out of end-of-turn mode
      self.SetMode(self.NOSELECTION, force=True)
      if self.current_player.control == self.HUMAN:
        # Human players: select the first unit
        self.SelectNextUnit() # sets self.mode to self.UNITSELECTED
        self.CentreViewOn(self.selected_unit.position)
        self.LoopGettingCommands()
      else:
        # Computer-controlled
        self.AI_TakeTurn()

    def LoopGettingCommands(self):
      """Allow the player to interact with the engine until an endgame or end-turn mode is reached.
      """
      while (self.mode not in [self.ENDOFTURN, self.VICTORY, self.DEFEAT]):
        self.GetCommand()

    def ResetFlagsAndMP(self):
      """Called at the start of each turn.
      """
      for thisunit in [u for u in self.units if u.controller == self.current_player]:
        if thisunit.can_act:
          thisunit.actions_left = thisunit.max_actions
        if thisunit.can_move:
          thisunit.MP = thisunit.max_MP

    def GetCommand(self):
      """Show any changes to the map, then let the user interact with the engine.
      """
      self.Show()
      self.is_in_interaction = True
      func = ui.interact()
      self.is_in_interaction = False
      if func is not None:
        func()

    # Certain mode transitions are prohibited in some circumstances, so mode is a property
    def SetMode(self, newMode, force=False):
      """
      Change the game mode to the specified mode. You can do this by just assigning to the UnitEngine's ''mode'' property, but if the ''mode'' is currently one of [UnitEngine.ENDOFTURN, UnitEngine.VICTORY, UnitEngine.DEFEAT], then the mode will not be changed unless the optional ''force'' parameter is True. Note that UnitEngine.MOVEMODE and UnitEngine.ACTMODE cannot be entered if the engine's selected_unit is None.
      """
      if newMode in [self.MOVEMODE, self.ACTMODE] and self.selected_unit is None:
        # We can't enter movement or action mode with no selected unit
        return
      if newMode == self.MOVEMODE and (self.selected_unit.position[0] is None or not self.selected_unit.can_move):
        # We can't move if we don't have a position, or if we're a unit that can't move.
        return
      if newMode == self.ACTMODE and not self.selected_unit.can_act:
        # We can't act if we're a unit that, well, can't act.
        return
      if self._mode in [self.ENDOFTURN, self.VICTORY, self.DEFEAT] and not force:
        # Only change our mode away from an endgame mode if force is true
        return
      self._mode = newMode
      self.HighlightNoSquares()
      if self.callbacks.mode_change is not None:
        self.callbacks.mode_change()

      if newMode == self.MOVEMODE:
        self.HighlightMoveableSquares()
      elif newMode == self.ACTMODE:
        self.EnterActionMode()

    CurriedSetMode = renpy.curry(SetMode)
    def GetMode(self):
      return self._mode
    mode = property(GetMode, SetMode)

    def CancelAction(self):
      """Cancel whatever's currently happening.
      If we're in Move or Act mode, return to Unit Selected mode. If we're in Unit Selected mode, select nothing. As a special case, if we've got nothing selected, then go to the game menu, as would normally happen for pressing Escape.
      Called by pressing Escape or clicking Cancel.
      """
      if self.mode == self.NOSELECTION:
        renpy.call_in_new_context("_game_menu")
      elif self.mode <= self.UNITSELECTED:
        self.SelectNothing()
      else:
        self.mode = self.UNITSELECTED

    # Method SelectNextUnit(): step through the units, starting
    # at g.selected_unit, to find one which can move or act.
    # i.e. MP mode: is_alive and (can_move and MP>0) or (can_act and actions_left>0)
    #      AP mode: is_alive and MP>0 and (can_move or can_act)
    def SelectNextUnit(self, keep_current=False):
      """Select the next unit which can do something.
      If the parameter ''keep_current'' is true, then the current unit will remain selected if there's anything it can do; otherwise, the next unit will be selected as usual.
      """
      if self.CheckEndGame():
        return
      if self.selected_unit and keep_current:
        # If we've got a selected unit and we're trying to keep it, start at that unit's index
        firstunitnum = self.units.index(self.selected_unit)
      elif self.selected_unit:
        # If we've got a selected unit and we're not trying to keep it, start at that unit's index + 1
        firstunitnum = self.units.index(self.selected_unit) + 1
        if firstunitnum >= len(self.units):
          firstunitnum -= len(self.units)
      else:
        # If we've got no selected unit, start at the first unit.
        firstunitnum = 0
      thisunitnum = firstunitnum
      continuelooping = True
      # Loop to find a unit that's both alive and has movement or action left
      # But don't loop endlessly: stop at firstunitnum
      while continuelooping:

        # We've got a unit to consider
        thisunit = self.units[thisunitnum]
        if not thisunit.controller is self.current_player: # keep looping if this unit isn't controlled by the current player
          continuelooping = True

        elif (not thisunit.is_alive): # keep looping if this unit's dead
          continuelooping = True

        elif (self.movement_style == self.MP and
              ((thisunit.MP == 0 or not thisunit.can_move) and
               (thisunit.actions_left == 0 or len(thisunit.actions) == 0))):
          # this unit can't do anything more
          continuelooping = True

        elif (self.movement_style == self.AP and
              (thisunit.AP == 0 or
               (not thisunit.can_move and len(thisunit.actions) == 0))):
          # this unit can't do anything more
          continuelooping = True

        else: # We've found a unit that can do something: stop here
          continuelooping = False
          self.SelectUnit(thisunit)

        # Carry on around the loop
        if continuelooping:
          thisunitnum += 1
          if thisunitnum >= len(self.units):
            thisunitnum -= len(self.units)
          if thisunitnum == firstunitnum:
            # stop looping if we're out of units
            continuelooping = False
            self.NoUnitToSelect()

      # SelectUnit() doesn't centre the screen on the selected unit by default,
      # but we do want to centre on selected unit if we've clicked "select next unit"
      if self.selected_unit is not None:
        self.CentreViewOn(self.selected_unit.position)


    def SelectUnit(self, unit):
      """Selects the specified unit.
      """
      self.selected_unit = unit
      self.mode = self.UNITSELECTED

    def SelectNothing(self):
      """Unselects whatever unit may be selected.
      """
      self.selected_unit = None
      self.mode = self.NOSELECTION

    def SelectCurrentOrNextUnit(self):
      """Select the current unit if it's got anything left it can do, or the next unit otherwise.
      """
      self.SelectNextUnit(keep_current = True)

    def NoUnitToSelect(self):
      """Called when we try to select the next unit that can do something, and discover no units can do anything.
      """
      if self.end_turn_when_finished:
        self.EndTurn()
      else:
        self.SelectNothing()

    @renpy.curry
    def ClickUnit(self, unit):
      """Process a user click on the specified unit. Curried, as we need to specify particular units when defining callbacks.
      """
      self.Cursor.position = unit.position
      self.ActivateCursorSquare()

    @renpy.curry
    def ClickSquare(self, (gx, gy)):
      """Process a user click on the specified square. Curried, as we need to specify particular squares when defining callbacks.
      """
      self.Cursor.position = (gx, gy)
      self.ActivateCursorSquare()

    def ActivateCursorSquare(self):
      """Choose or select the current square.
      Called when the user clicks a square or presses Enter on it.
      If there's a highlight on that square, call the highlight's clicked_function. If not, select what's in this square.
      """
      thisMapSquare = map[self.Cursor.position[0]][self.Cursor.position[1]]
      # Was the square highlighted as a possible target for move or action?
      if thisMapSquare.highlight and thisMapSquare.clicked_function:
        # If so, then call the clicked_function, passing the position as its argument
        thisMapSquare.clicked_function(self.Cursor.position)
      # If not, then select what's in the square if we're in an appropriate mode
      elif self.mode in [self.NOSELECTION, self.UNITSELECTED]:
        if thisMapSquare.unit:
          # If the unit has a non-default clicked function, call that
          if thisMapSquare.unit.nondefault_clicked_function:
            # Need to call it twice, as clicked functions are wrapped in ui.returns
            func = thisMapSquare.unit.clicked()
            if func is not None:
              func()
          else:
            # If the clicked function is the default function, then it's how we got in here
            # So just select that unit
            self.SelectUnit(thisMapSquare.unit)
        else:
          self.SelectNothing()

    def ConfirmEndTurn(self):
      """Pop up a menu asking the player if they really want to end their turn.
      """
      if self.confirm_end_turn:
        self.OverlaidMenu((("End Turn", self.EndTurn),
                         ("Cancel", ui.returns(False))),
                        cancel_button = False)
      else:
        self.EndTurn()

    def EndTurn(self):
      """End the current player's turn.
      """
      self.mode = self.ENDOFTURN

    def MoveUnitToSquare(self, position):
      """Move the selected unit to the specified square.
      """
      self.selected_unit.MoveToSquare(position)
      self.SelectCurrentOrNextUnit()

    ### Highlighting squares: ###
    def HighlightNoSquares(self):
      """Remove all highlighting from the map.
      """
      self.HighlightSquares(lambda p: False, None)

    def HighlightMoveableSquares(self):
     if self.stored_movement_costs_for_unit is not self.selected_unit:
        self.CalculateMovementCosts(self.selected_unit)
     # Now display highlights on the squares which are moveable-to
     self.HighlightSquares(criterion=self.MoveableSquareCriterion,
                           clicked_function=self.MoveUnitToSquare,
                           highlight=self.move_square_highlight)

    def MoveableSquareCriterion(self, (gx, gy)):
      """Return True if the current unit can move to the specified square.

      The "current" unit is the unit for which CalculateMovementCosts was most recently called.
      """
      # Can we move here?
      if self.map[gx][gy].StoredMovementCost > 0 and \
         self.map[gx][gy].StoredMovementCost <= self.selected_unit.MP and \
         self.map[gx][gy].unit is None:
        # Yes, we can: tag the square as moveable
        return True
      else:
        # No, we can't
        return False

    def HighlightSquares(self, criterion, highlight='___unspecified', clicked_function=None, blank_others = True):
      """Highlight some map squares with a sprite, and make clicks on them call a function.

      Parameters:
        criterion: A function which takes a (gx,gy) tuple, and should return true or false. Required.
        highlight: A string (image file name) or a pair of strings (normal image, hovered image). Default is the engine's act_square_highlight.
        clicked_function: Will be called with a (gx,gy) tuple after ui.interact. Default is None.
        blank_others: If True (default), all squares which don't meet the criterion will have their highlights removed. If False, they won't be modified, and only the squares which do meet the criterion will have their highlight changed.
      """
      if highlight == '___unspecified':
        highlight = self.act_square_highlight

      # On the squares which fit the criterion,
      # Display highlights and store the clickedfunction
      for gx in range(self.gxmax):
        for gy in range(self.gymax):
          if criterion((gx, gy)):
            self.map[gx][gy].highlight = highlight
            self.map[gx][gy].clicked_function = clicked_function
          elif blank_others:
            # If we don't match, blank the highlight if we've been told to
            self.map[gx][gy].highlight = None
            self.map[gx][gy].clicked_function = None


    def CalculateMovementCosts(self, unit, unitMPtotal = None):
      """Calculate how many MP the unit will take to move to each square of the map.

      The cost is stored in each map square's .StoredMovementCost property.
      """
      if unitMPtotal is None:
       unitMPtotal = unit.MP
      (unitx, unity) = unit.position
      # blank the stored MoveCosts
      for x in range(self.gxmax):
        for y in range(self.gymax):
          self.map[x][y].StoredMovementCost = -1
          self.map[x][y].StoredMovementCostSupposingReachable = -1

      # Note that we're storing movement costs for this unit
      self.stored_movement_costs_for_unit = unit

      # Build up contours of each movement cost
      minx = unitx-1; miny = unity-1
      maxx = unitx+1; maxy = unity+1
      self.map[unitx][unity].StoredMovementCost = 0
      for contour in range(1, unitMPtotal+1):
        somethingToDo = False
        for x in range(minx, maxx+1):
          for y in range(miny, maxy+1):
            if not self.InBounds((x,y)):
              continue
            if self.map[x][y].StoredMovementCost != -1:
              continue
            if not self.map[x][y].terrain_type.MoveableThrough:
              continue
            thiscost = self.map[x][y].terrain_type.MoveCost
            # Look for an adjacent square that we've previously marked that we can move to
            # for a cost of (contour - thiscost)
            if self.movement_directions == 4:
              dirstocheck = [(-1,0), (+1,0), (0,-1), (0,+1)]
            else:
              dirstocheck = [(-1,0), (+1,0), (0,-1), (0,+1), (-1,-1), (-1,+1), (+1,-1), (+1,+1)]

            for (dx, dy) in dirstocheck:
              thisx = x + dx
              thisy = y + dy
              if self.InBounds((thisx, thisy)):
                if self.map[thisx][thisy].StoredMovementCost != -1:
                  # There's a not-yet-marked square adjacent to a square we can move to:
                  # that means there's reason to keep looping
                  somethingToDo = True
                  if self.map[thisx][thisy].StoredMovementCost == contour - thiscost:
                    # Add this square to the contour
                    self.map[x][y].StoredMovementCostSupposingReachable = contour
                    if self.allow_moving_through_units or self.map[x][y].unit is None:
                      self.map[x][y].StoredMovementCost = contour
                    if x==minx:
                      minx-=1
                    elif x==maxx:
                      maxx+=1
                    if y==miny:
                      miny-=1
                    elif y==maxy:
                      maxy+=1
                    break
        if not somethingToDo:
          break


    def EnterActionMode(self):
      if len(self.selected_unit.actions) == 0:
        # Can't enter action mode with 0 actions
        self.mode = self.UNITSELECTED
      elif len(self.selected_unit.actions) == 1:
        # Call the one action function
        self.selected_unit.actions.values()[0]()
      else:
        # Supply a menu of the actions
        self.OverlaidMenu(self.selected_unit.actions.items())


    def CheckEndGame(self):
      """Check if all the player's units are dead, or (if the engine's ''end_when_all_enemies_dead'' parameter is True) all enemies are dead. The engine's ''mode'' will be set to e.VICTORY or e.DEFEAT if appropriate. CheckEndGame itself returns True if the end-game conditions are met, or False if not.
      """
      PlayerUnits = [u for u in self.units if u.controller is self.PLAYER]
      EnemyUnits  = [u for u in self.units if u.controller is not self.PLAYER]
      if len(PlayerUnits) == 0:
        self.mode = self.DEFEAT
      elif len(EnemyUnits) == 0 and self.end_when_all_enemies_dead:
        self.mode = self.VICTORY
      if self.mode in [self.DEFEAT, self.VICTORY]:
        return True
      else:
        return False

    def IsInInteraction(self):
      """Return True if the engine's in the middle of its own ui.interact call.
      Clicked functions should condition on this before doing anything significant, so as to not also trigger if the item's clicked during a VN segment or similar. Example:
        clicked = e.CurriedConditionallyReturn(e, e.IsInInteraction, myFunction)
      """
      return self.is_in_interaction

    def ConditionallyReturn(self, conditionfunction, value):
      """Return value only if the conditionfunction evaluates to true.
      The curried version is called CurriedConditionallyReturn.
      """
      if conditionfunction():
        return value
      else:
        return
    CurriedConditionallyReturn = renpy.curry(ConditionallyReturn)

    ############################### OVERLAYS ###############################

    ## Overlay for the buttons:

    def overlay_ShowButtons(self):
      if self.mode>0:
        self.ShowButtons()

    def ShowButtons(self):

      if self.menu_items:
        # Don't show the normal buttons if we've got a popup menu
        ui.frame(style=style.engine_frame["menu"])
        ui.menu(style=style.engine_menu, menuitems=self.menu_items)

      elif self.show_buttons:
        # Normal buttons
        ui.frame(style=style.engine_frame["buttons"])

        ui.vbox() # The different rows of buttons
        ui.hbox() # The row of Next Unit and Next Turn buttons
        engine_button("Next Unit", clicked=ui.returns(self.SelectNextUnit))
        engine_button("End Turn", clicked=ui.returns(self.ConfirmEndTurn))
        ui.close() # End of the Next Unit / End Turn row
        ui.hbox() # The row of Move and Act buttons
        showMoveButton = (self.mode >= self.UNITSELECTED) and self.selected_unit.controller == self.current_player
        showActButton = showMoveButton
        if showMoveButton:
          enableMoveButton = (self.mode != self.MOVEMODE) and self.selected_unit.MP > 0 and self.selected_unit.can_move
          if enableMoveButton:
            clicked = ui.returns(self.CurriedSetMode(self, self.MOVEMODE))
          else:
            clicked = None
          engine_button("Move", clicked=clicked)
        if showActButton:
          enableActButton = (self.mode != self.ACTMODE and
            (self.movement_style == self.MP and
              (self.selected_unit.actions_left > 0 and self.selected_unit.can_act)) or
            (self.movement_style == self.AP and
              (self.selected_unit.AP >= 0 and self.selected_unit.can_act))
                               )
            # *** Todo: This should be a better criterion and/or more customisable
          if enableActButton:
            clicked = ui.returns(self.CurriedSetMode(self, self.ACTMODE))
          else:
            clicked = None
          engine_button("Act", clicked=clicked)
        ui.close() # End of the Move / Act row
        showCancelButton = (self.mode != self.NOSELECTION)
        if showCancelButton:
          if self.mode == self.MOVEMODE:
            cancelButtonText = "Cancel move"
          elif self.mode == self.ACTMODE:
            cancelButtonText = "Cancel action"
          else:
            cancelButtonText = "Cancel"
          engine_button(cancelButtonText, clicked=ui.returns(self.CancelAction))
        ui.close() # End of the vbox


    # Overlay to show unit info

    def overlay_ShowUnitInfo(self):
      if (self.mode in [self.UNITSELECTED, self.MOVEMODE, self.ACTMODE]) and self.show_unit_info:
        self.ShowUnitInfoPanel()

    def ShowUnitInfoPanel(self):
      ui.frame(style=style.engine_frame["unit info"])
      ui.vbox() # Main vertical layout
      for thisFunc in self.unit_info:
        if callable(thisFunc):
          # If thisFunc is callable itself, it's an unconditional item
          thisFunc()
        else:
          # If thisFunc is actually a tuple, then the first element is
          # the condition of whether or not to show this element, and
          # the second is the callable
          if thisFunc[0]():
            thisFunc[1]()
      ui.close() # End of unit info panel

    def ShowUnitName(self):
      """Create a ui label showing the selected unit's name, suitable for inclusion in e.unit_info.
      The window will be in style style.engine_label["unit name"], and the text in style style.engine_label_text["unit name"].
      """
      ui.window(style=style.engine_label["unit name"])
      ui.text(self.selected_unit.name, style=style.engine_label_text["unit name"])

    def ShowUnitHP(self):
      """Create a ui label showing the selected unit's HP, suitable for inclusion in e.unit_info.
      The window will be in style style.engine_label["HP"], and the text in style style.engine_label_text["HP"].
      """
      ui.window(style=style.engine_label["HP"])
      HPtext = "%s: %d/%d" % (_("HP"), self.selected_unit.HP, self.selected_unit.max_HP)
      ui.text(HPtext, style=style.engine_label_text["HP"])

    def ShowUnitMP(self):
      """Create a ui label showing the selected unit's MP, suitable for inclusion in e.unit_info.
      The window will be in style style.engine_label["MP"], and the text in style style.engine_label_text["MP"].
      """
      ui.window(style=style.engine_label["MP"])
      MPtext = "%s: %d/%d" % (_("MP"), self.selected_unit.MP, self.selected_unit.max_MP)
      ui.text(MPtext, style=style.engine_label_text["MP"])

    def ShowUnitAP(self):
      """Create a ui label showing the selected unit's AP, suitable for inclusion in e.unit_info.
      The window will be in style style.engine_label["AP"], and the text in style style.engine_label_text["AP"].
      """
      ui.window(style=style.engine_label["AP"])
      APtext = "%s: %d/%d" % (_("AP"), self.selected_unit.AP, self.selected_unit.max_AP)
      ui.text(APtext, style=style.engine_label_text["AP"])

    def ShowUnitActionsLeft(self):
      """Create a ui label showing the selected unit's number of MP-mode actions left, suitable for inclusion in e.unit_info.
      The window will be in style style.engine_label["actions left"], and the text in style style.engine_label_text["actions left"].
      """
      ui.window(style=style.engine_label["actions left"])
      actlefttext = "%s: %d/%d" % (_("Actions left"), self.selected_unit.actions_left, self.selected_unit.max_actions)
      ui.text(actlefttext, style=style.engine_label_text["actions left"])

    def ShowUnitPosition(self):
      """Create a ui label showing the selected unit's (gridx, gridy) position, suitable for inclusion in e.unit_info.
      The window will be in style style.engine_label["position"], and the text in style style.engine_label_text["position"].
      """
      ui.window(style=style.engine_label["position"])
      postext = "(%d, %d)" % self.selected_unit.position
      ui.text(_(postext), style=style.engine_label_text["position"])

     ############################## AI FUNCTIONS ##############################

    def AI_TakeTurn(self):
      """A naive and stupid AI.

      This is an extremely simplistic AI. You'll probably want to write your own instead.
      It takes the turn for e.current_player. It loops through each of that player's units in the order they were created. For each one, it tries to move as close to a unit controlled by e.PLAYER as it can, planning a route up to 4 turns ahead. Then it calls the first function in u.AttackFunctions, supplying the targeted unit as a parameter.
      No consideration is given to weapon ranges, strategic considerations, or anything like that. It might do for attacks by melee-fighting beasts.
      """
      # For each AI unit:
      for thisUnit in [u for u in self.units
                       if u.controller is self.current_player]:
        self.SelectUnit(thisUnit)
        mygx, mygy = thisUnit.position
        # Centre the view on it
        self.CentreViewOn(thisUnit.position)
        self.Show()
        # Find the closest enemy unit - reachable within 4 turns
        self.CalculateMovementCosts(thisUnit, thisUnit.MP * 4)
        bestUnit = None
        bestCost = 9999999
        enemyUnits = [u for u in self.units
                       if u.controller is not self.current_player]
        for u in enemyUnits:
          egx, egy = u.position
          #xstep = sign(egx - mygx)
          #ystep = sign(egy - mygy)

          thisCost = self.map[u.position[0]][u.position[1]].StoredMovementCostSupposingReachable
          if thisCost != -1 and thisCost <= bestCost:
            bestUnit = u
            bestCost = thisCost

        # If we can't find one, this unit does nothing: onto the next unit
        if bestUnit is None:
          continue
        # Store the target unit, so the attack callbacks can know what to hit
        self.selected_unit.target_unit = bestUnit

        # Move towards it
        # MoveToSquare will only move as far towards the target as the moving unit
        # has movement points, so this does what we want
        thisUnit.MoveToSquare(self.selected_unit.target_unit.position)

        # Attack it if possible
        # Choose a random action
        chosenAction = renpy.random.choice(self.selected_unit.actions.values())
        chosenAction()


    ######################### FUNCTIONS FOR USER USE ##########################
    # These functions aren't called by any TileEngine or UnitEngine code, but are provided for user use.
    # It's expected that most user callbacks (mode_change, unit_moves or unit_acts) will use functions like these.
    # The first few are criteria, suitable for passing to HighlightSquares either on their own or combined.

    def ContainsAnyUnit(self, (thisx, thisy)):
      """Return True if the specified square contains any unit.
      This criterion is suitable for use in a UnitEngine's HighlightSquares function.
      """
      return (self.map[thisx][thisy].unit is not None)

    def ContainsPlayerUnit(self, (thisx, thisy)):
      """Return True if the specified square contains a unit controlled by the current player.
      This criterion is suitable for use in a UnitEngine's HighlightSquares function.
      """
      return (self.map[thisx][thisy].unit is not None and
          self.map[thisx][thisy].unit.controller is self.current_player)

    def ContainsNonPlayerUnit(self, (thisx, thisy)):
      """Return True if the specified square contains a unit controlled by someone other than the current player.
      This criterion is suitable for use in a UnitEngine's HighlightSquares function.
      """
      return (self.map[thisx][thisy].unit is not None and
          self.map[thisx][thisy].unit.controller is not self.current_player)

    def WithinNSpacesOf(self, n, (thisx, thisy)):
      """Return True if the specified square is within the specified number of orthogonal steps from the selected unit.
      This criterion is suitable for use in a UnitEngine's HighlightSquares function.
      """
      (unitx, unity) = self.selected_unit.position
      dist = abs(thisx - unitx) + abs(thisy - unity)
      return (dist <= n)

    # IsPlayerUnit and IsNonPlayerUnit are criteria suitable for e.unit_info
    def IsPlayerUnit(self):
      """Return True if the selected unit is controlled by the current player.
      This criterion is suitable for use in a UnitEngine's ''unit_info'' property.
      """
      return (self.selected_unit is not None and self.selected_unit.controller is self.current_player)

    def IsNonPlayerUnit(self):
      """Return True if the selected unit is controlled by someone other the current player.
      This criterion is suitable for use in a UnitEngine's ''unit_info'' property.
      """
      return (self.selected_unit is not None and self.selected_unit.controller is not self.current_player)

    # Function OverlaidMenu
    def OverlaidMenu(self, menuitems, cancel_button=True):
      """Replace the normal buttons with a menu offering the specified options.
      This can be used to provide action options when the user selects "Act", or at most other times. Do not call it from inside a Show callback.
      Parameters:
        menuitems: A list of (string, callback) tuples, each one corresponding to a menu item to be shown.
        cancel_button: If True (default), a button will be added at the end of the list, which will call e.CancelAction() when selected.
      """
      if cancel_button:
        menuitems = copy.copy(menuitems)
        menuitems.append(('Cancel', self.CancelAction))
      self.menu_items = [(string, self.DismissMenu(self, callback)) for (string, callback) in menuitems]

    @renpy.curry
    def DismissMenu(self, callback):
      self.menu_items = None
      callback()


    # DestroyUnit, DamageUnit and DamageUnitBy(n) may be useful for ClickedFunctions provided to HighlightSquares.
    def DestroyUnit(self, unit):
      """Destroy the specified unit.
      The unit's ''is_alive'' flag is set to False, then the engine's ''callbacks.unit_destroyed'' function is called. Finally the unit is removed from  the engine's ''units'' list and the map.
      """
      unit.is_alive = False
      if self.callbacks.unit_destroyed:
        self.callbacks.unit_destroyed(unit=unit)
      unit.Remove()
      self.Show()

    def DamageUnit(self, damage, unit):
      """Reduce the specified unit's HP by the specified amount, destroying it if the HP is now 0 or less.
      Parameters:
        damage: The amount of damage to deal.
        unit: The unit to deal damage to.

      DamageUnitBy is the curried version. So for example, DamageUnitBy(30) will return a function that, when called with some unit, deals 30 damage to that unit.
      """
      unit.HP -= damage
      if self.callbacks.unit_damaged:
        self.callbacks.unit_damaged(unit=unit, damage=damage)
      if unit.HP <= 0:
        self.DestroyUnit(unit)

    DamageUnitBy = renpy.curry(DamageUnit)

    # FinishAction should be called when your action's finished.
    def FinishAction(self, APCost=1, actions_consumed=1):
      """End processing of an action and return to UNITSELECTED mode. This should be called from the end of every user action function, once the action has completely finished, to return to normal operation.
      Parameters:
        APCost: If the engine is in AP mode, the selected unit's AP will be reduced by this amount. Default is 1.
        actions_consumed: If the engine is in MP mode, the selected unit's number of remaining actions will be reduced by this amount. Default is 1.
      """
      # Spend the action / AP
      if self.movement_style == self.MP:
        self.selected_unit.actions_left -= actions_consumed
      else:
        self.selected_unit.AP -= APCost
      # Return to UnitSelected mode
      self.SelectCurrentOrNextUnit()


    #### End of class definition of UnitEngine



  #################################
  ### Definition of Unit object ###
  #################################
  class Unit(Sprite, Struct):
    def __init__(self,
        # Required keyword parameters
        engine = None,
        name = None,
        sprite_base_name = None,
        # Optional keyword parameters
        hover_sprite_base_name = None, # defaults to sprite_base_name
        max_HP = None, max_MP = None, max_AP = None, max_actions = 1,
        direction_facing = None,       # defaults to engine.DIR_NEUTRAL
        position = (None, None),
        can_move = True, can_act = True,
        controller = None,             # defaults to engine.PLAYER
        display_directions = None,     # defaults to engine.display_directions
        actions = {},
        clicked = -1,                  # -1 means default clicked function; None means not clickable
        sprite_status = "",
        **properties):
      """Create a Unit object. This should never be called directly: the parent UnitEngine's Unit() function should be called instead.
      """

      # First, check we've got the required parameters
      if engine is None:
        raise Exception, "Unit() must be created by a UnitEngine's Unit() function!"
      if sprite_base_name is None:
        raise Exception, "Unit requires a sprite_base_name!"

      # Store the properties specified
      self.engine = engine
      self.max_MP = max_MP  # max_HP is stored later
      self.max_AP = max_AP
      self.can_move = can_move
      self.can_act = can_act
      self.actions = actions
      self.max_actions = max_actions

      # Store properties which use property objects
      self._pos = position
      self._sprite_base_name = sprite_base_name
      self._sprite_status = sprite_status

      # Store properties dependent on inputs
      self.name = name or sprite_base_name
      self._direction_facing = direction_facing or self.engine.DIR_NEUTRAL
      self.controller = controller or engine.PLAYER
      self.display_directions = display_directions or engine.display_directions

      assert self.controller in self.engine.player_list, "Unit %s's controller is not in UnitEngine's player_list!" % name
      if hover_sprite_base_name is None:
        self._hover_sprite_base_name = self._sprite_base_name
      else:
        self._hover_sprite_base_name = hover_sprite_base_name
      self.      base_name_dot_index = self.      sprite_base_name.rfind(".")
      self.hover_base_name_dot_index = self.hover_sprite_base_name.rfind(".")

      if max_HP is not None:
        thisHP_scale = self.controller.HP_scale
        self.max_HP = int(max_HP * thisHP_scale)
      else:
        self.max_HP = None

      # Assign default values to other properties
      self.is_alive = True
      self.actions_left = self.max_actions
      self.MP = self.max_MP
      self.AP = self.max_AP

      # Register ourselves with the UnitEngine
      self.engine.units.append(self)

      # Store us on the map, if our position's defined
      (gx, gy) = position
      if gx is not None and gy is not None:
        assert self.engine.map[gx][gy].unit is None, "Adding a unit to (%d,%d) when there's already a unit there!" % position
        self.engine.map[gx][gy].unit = self

      # Find our number
      unitnum = len(self.engine.units)

      # Decide the sprite name and clickedfunction
      self.UpdateSpriteName()

      # Create the Sprite
      Sprite.__init__(self, engine=engine, sprite_name=self.sprite_name, position=self.position, clicked=clicked)

      # Assign the clicked parameter
      self.clicked = clicked

      # Process unhandled properties
      Struct.__init__(self, **properties)


    # Properties that do special things on setting:
    # When the user sets u.position, update the engine's map[] references
    def SetPos(self, val):
      self.OldPos = self._pos
      # Only consider the oldpos if it was non-None
      if self.OldPos[0]:
      # If the old position had any units, pop one out into its .unit field
        oldMapSquare = self.engine.map[self.position[0]][self.position[1]]
        if oldMapSquare.under_units:
          oldMapSquare.unit = oldMapSquare.under_units.pop()
        else:
          oldMapSquare.unit = None
      # Update the position
      self._pos = val
      # Only consider the new pos if it's non-None
      if val[0]:
        # If the new position has any units, push it into .under_units
        newMapSquare = self.engine.map[self.position[0]][self.position[1]]
        if newMapSquare.unit is not None:
          newMapSquare.under_units.append(newMapSquare.unit)
        # Store ourselves in the .unit
        newMapSquare.unit = self

    def GetPos(self):
      return self._pos
    position = property(GetPos, SetPos)

    # AP is an alias for MP
    def Set_AP(self, val):
      self.MP = val
    def Get_AP(self):
      return self.MP
    AP = property(Get_AP, Set_AP)

    # When the user sets u.sprite_status, u.direction_facing or u.sprite_base_name, update the sprite_name
    def Set_sprite_status(self, val):
      self._sprite_status = val
      self.UpdateSpriteName()
    def Get_sprite_status(self):
      return self._sprite_status
    sprite_status = property(Get_sprite_status, Set_sprite_status)
    def Set_direction_facing(self, val):
      self._direction_facing = val
      self.UpdateSpriteName()
    def Get_direction_facing(self):
      return self._direction_facing
    direction_facing = property(Get_direction_facing, Set_direction_facing)
    def Set_sprite_base_name(self, val):
      self._sprite_base_name = val
      self.UpdateSpriteName()
    def Get_sprite_base_name(self):
      return self._sprite_base_name
    sprite_base_name = property(Get_sprite_base_name, Set_sprite_base_name)
    def Set_hover_sprite_base_name(self, val):
      self._hover_sprite_base_name = val
      self.UpdateSpriteName()
    def Get_hover_sprite_base_name(self):
      return self._hover_sprite_base_name
    hover_sprite_base_name = property(Get_hover_sprite_base_name, Set_hover_sprite_base_name)

    # When the user sets u.clicked, note whether it's to the default or not
    def Set_clicked(self, val):
      if val == -1:
        # Special value of -1 to reset the clicked function
        self.nondefault_clicked_function = False
        self._clicked = ui.returns(self.engine.ClickUnit(self.engine, self))
      else:
        self.nondefault_clicked_function = True
        self._clicked = val
    def Get_clicked(self):
      return self._clicked
    clicked = property(Get_clicked, Set_clicked)

    def UpdateSpriteName(self):
      # For the sprite_name, take everything up to the dot...
      self.sprite_name = self.sprite_base_name[:self.base_name_dot_index]
      # ...then append "-" and the direction, if it's non-neutral...
      if self.direction_facing.name:
        self.sprite_name += "-" + self.direction_facing.name
      # ...then "-" and the status, if that's nonempty...
      if self.sprite_status:
        self.sprite_name += "-" + self.sprite_status
      # ...and finally append the dot and extension
      self.sprite_name += self.sprite_base_name[self.base_name_dot_index:]

      # Now do the same for the hover_sprite_name
      self.hover_sprite_name = self.hover_sprite_base_name[:self.hover_base_name_dot_index]
      if self.direction_facing.name:
        self.hover_sprite_name += "-" + self.direction_facing.name
      if self.sprite_status:
        self.hover_sprite_name += "-" + self.sprite_status
      self.hover_sprite_name += self.hover_sprite_base_name[self.base_name_dot_index:]

    def TurnToFace(self, target):
      """Turn the unit to point towards the target.
      The unit's ''direction_facing'' property will be updated. The number of possible values is the unit's ''display_directions'' property.
      Parameters:
        target: Can be either a unit, or a (gridx, gridy) tuple.
      """
      if self.display_directions == 1:
        # This unit doesn't rotate
        return
      # We accept either a unit or a (x,y) position. We just want the position.
      if hasattr(target, "position"):
        target = target.position
      # Determine the dx and dy
      myx, myy = self.position
      tx, ty = target
      dx = tx - myx
      dy = ty - myy
      sdx = sign(dx)
      sdy = sign(dy)
      # See whether to ignore one or other of them, and which one
      adx = abs(dx)
      ady = abs(dy)
      if self.display_directions == 4:
        # Strip out the smaller of dx and dy
        if adx >= ady:
          sdy = 0
        else:
          sdx = 0
      else:
        # If one delta is twice the size of the other, strip out the smaller
        if adx>=2*ady:
          sdy=0
        elif ady>=2*adx:
          sdx=0
      self.direction_facing = self.engine.Direction((sdx, sdy))

    def MoveToSquare(self, position, follow=True, spend_MP=True, transition=move):
      """Find a path for the unit from its current position to the specified position and move it.
      The unit's ''position'' and ''direction_facing'' are updated at each step, and the engine's ''callbacks.unit_moves'' will be called.
      Parameters:
        follow: If True (default), the UnitEngine's viewport will be scrolled to centre on the unit at each step.
        spend_MP: If True (default), the unit's MP/AP total will be decreased at each step by the grid square's ''movement_cost''.
      """
      # *** TODO:
      # To allow Flying objects (or boats etc), each Unit has a list of TerrainTypes it can move through.

      # Blank the highlights
      self.engine.HighlightNoSquares()

      # Plan the route from our position to position
      steps = self.CalculateRoute(position)

      # If we're spending MP, cut off at the highest point we can afford
      if spend_MP:
        while len(steps) > 0 and self.engine.map[steps[-1][0]][steps[-1][1]].StoredMovementCost > self.MP:
          steps.pop()

      # Disallow steps while there's a unit in the last one
      while len(steps) > 0 and self.engine.map[steps[-1][0]][steps[-1][1]].unit is not None:
          steps.pop()

      # Loop over the steps
      for s in steps:
        # If we're spending MP, calculate to see if we can
        if spend_MP:
          # Calculate move cost to move this step
          thisStepCost = map[s[0]][s[1]].terrain_type.MoveCost
          # If we can't afford it, break out of the for loop
          if thisStepCost > self.MP:
            break
          # Otherwise, subtract the MP
          self.MP -= thisStepCost

        # Move the one step
        self.MoveOneStep(s, follow)
        # Invalidate the stored movement costs, as they'll now not apply
        self.engine.stored_movement_costs_for_unit = None

      # Once we're done moving, select the static image
      self.sprite_status = ""
      # Re-show
      self.engine.Show()

    def MoveOneStep(self, NewPos, follow=False, transition=move):
      # Update the direction_facing
      self.TurnToFace(NewPos)

      # Set the animation image
      self.sprite_status = self.engine.moving_sprite_status
      # Update our position
      self.MoveTo(NewPos)

      # Tell the engine we've moved
      # Centre the view on the new position, if follow is true
      if follow:
        self.engine.CentreViewOn(NewPos)
      # Re-show
      self.engine.Show()
      # Pause for the movement
      renpy.pause(self.engine.movement_duration)
      # Call the callback
      if self.engine.callbacks.unit_moves:
        self.engine.callbacks.unit_moves(self, NewPos)

    def CalculateRoute(self, targetpos):
      """Return a path for the unit from its current position to the specified position.
      Output is a list of (gridx, gridy) positions.
      Parameters:
        targetpos: The target position, as a (gridx, gridy) tuple.
      """
      if self.engine.stored_movement_costs_for_unit is not self:
        self.engine.CalculateMovementCosts(self)
      # Now we know that the map's stored movement costs are for us
      steps = [ ]
      (thisX, thisY) = targetpos
      # Loop until we get home
      while (thisX, thisY) != self.position:
        # Push this step onto the end of the steps list
        steps.append((thisX, thisY))
        # Build a list of squares to check, from most preferred to least.
        # If there's an orthogonal step and a diagonal step that are equally good, prefer the orthogonal.
        # If there are two orthogonal steps that're equally good, prefer the one matching the direction in which we have further to go.
        remainingx = abs(thisX - self.position[0])
        remainingy = abs(thisY - self.position[1])
        if remainingx >= remainingy:
          dirstocheck = [(-1,0), (+1,0), (0,-1), (0,+1)]
        else:
          dirstocheck = [(0,-1), (0,+1), (-1,0), (+1,0)]
        if self.engine.movement_directions == 8:
          dirstocheck.extend([(-1,-1), (-1,+1), (+1,-1), (+1,+1)])

        # Check in each of the 4/8 directions, looking for the square with minimum StoredMovementCost
        bestCost = map[thisX][thisY].StoredMovementCost
        if bestCost == -1:
          # We won't be able to move all the way to the target, but we might be able to move next to it
          bestCost = 99999
        bestdx = 0
        bestdy = 0
        for (dx,dy) in dirstocheck:
          if self.engine.InBounds((thisX+dx, thisY+dy)):
            if map[thisX+dx][thisY+dy].StoredMovementCost != -1 and map[thisX+dx][thisY+dy].StoredMovementCost < bestCost:
              (bestdx, bestdy) = (dx, dy)
              bestCost = map[thisX+dx][thisY+dy].StoredMovementCost
        # If we couldn't find one, the destination's unreachable
        if bestdx == 0 and bestdy == 0:
          break
        # Once we've found the cheapest direction, take it and carry on around the loop
        thisX += bestdx
        thisY += bestdy

      # Now, we've got a list of steps, starting at targetpos, going to the unit's current position.
      # We want it the other way round.
      steps.reverse()
      return steps

    def Remove(self):
      """Permanently remove the unit from the UnitEngine.
      """
      self.engine.map[self.position[0]][self.position[1]].unit = None
      self.engine.units.remove(self)
      Sprite.Remove(self)
